<?php

/**
 * This plugin allows to add access records to customers
 * ordered configured products by schedule. For example:
 *   - from 7-th day to 14-th day add access to Product#1
 *   - from 1-th Friday to 3-th Sunday add access to Product #2
 *   - from 1-th day of month add access to Product #3
 * @am_plugin_api 6.0
 */
class Am_Plugin_ScheduleAccess extends Am_Plugin
{
    const PLUGIN_STATUS = self::STATUS_BETA;
    const PLUGIN_COMM = self::COMM_COMMERCIAL;
    const PLUGIN_REVISION = '6.3.29';

    const START = 'start';
    const FOREVER = 'forever';
    const EXPIRE = 'expire';

    function onGridProductInitForm(Am_Event $event)
    {
        $event->getGrid()->getForm()
            ->addElement(new Am_Form_Element_ScheduleAccess('_schedule_access'))
            ->setLabel('Schedule Access');
    }

    function onGridProductBeforeSave(Am_Event $event)
    {
        $product = $event->getGrid()->getRecord();
        $val = $event->getGrid()->getForm()->getValue();
        $product->data()->setBlob('schedule_access', $val['_schedule_access']);
    }

    function onGridProductValuesToForm(Am_Event $event)
    {
        $args = $event->getArgs();
        $values = $args[0];
        $product = $event->getGrid()->getRecord();
        if ($schedule_access = $product->data()->getBlob('schedule_access'))
        {
            $values['_schedule_access'] = $schedule_access;
            $event->setArg(0, $values);
        }
    }

    function isEventBased($point)
    {
        return in_array($point, [self::START, self::EXPIRE]) || preg_match('/^[0-9]+-p$/', $point);
    }

    function onInitFinished(Am_Event $e)
    {
        //it should be after checkAllSubscriptions
        $this->getDi()->hook->add(Am_Event::DAILY, [$this, '_onDaily']);
    }

    function _onDaily(Am_Event $event)
    {
        $dateNow = $this->getDi()->dateTime;
        $time = amstrtotime($event->getDate());
        $dateNow->setDate(date('Y', $time), date('m', $time), date('d', $time));

        $products = $this->getDi()->productTable->selectObjects("SELECT t.*
            FROM ?_product t
                LEFT JOIN ?_data d ON (d.`table` = 'product' AND d.`id` = t.product_id AND d.`key`=?)
            WHERE d.`blob`<>?", 'schedule_access', '');

        foreach ($products as $product)
        {
            $schedule_access = json_decode($product->data()->getBlob('schedule_access'), true);
            foreach ($schedule_access as $product_id => $dates)
            {
                if ($product->pk() == $product_id)
                    continue;

                if ($this->isEventBased($dates['from'])) continue; //these cases handled separately on special hooks

                if (empty($dates['from']) || empty($dates['to']))
                    continue;

                try {
                    [$start, $stop] = $this->getSuitableStartInterval($dateNow, $dates['from'], $product->pk());
                } catch(Am_Exception_Configuration $ex) {
                    $this->getDi()->errorLogTable->logException($ex);
                    continue;
                }
                if (preg_match('/^([0-9]+)-d$/', $dates['from'], $_)) {
                    $access = $this->getDi()->accessTable->selectObjects(<<<CUT
                        SELECT a.user_id,
                               MAX(a.begin_date) AS begin_date,
                               MAX(a.expire_date) AS expire_date,
                               MAX(a.access_id) AS access_id,
                               a.product_id
                            FROM ?_access a
                            LEFT JOIN ?_access_cache ac ON ac.fn='product_id' AND ac.id=a.product_id AND ac.user_id = a.user_id
                            WHERE a.product_id=? AND ac.days=? AND ac.status=?
                            GROUP BY a.user_id
                        CUT,
                        $product->pk(),
                        $_[1],
                        'active'
                    );
                } else {
                    $access = $this->getDi()->accessTable->selectObjects(
                        "SELECT * FROM ?_access WHERE begin_date>=? AND begin_date<=? AND product_id=?",
                        $start,
                        $stop,
                        $product->pk()
                    );
                }
                foreach ($access as $a) {
                    $record = $this->getDi()->accessRecord;
                    $record->user_id = $a->user_id;
                    $record->begin_date = $dateNow->format('Y-m-d');
                    $record->expire_date = $this->getExpire($dateNow, $a->begin_date, $a->expire_date, $dates['to']);
//                    $record->invoice_id = $a->invoice_id;
//                    $record->invoice_public_id = $a->invoice_public_id;
//                    $record->invoice_payment_id = $a->invoice_payment_id;
//                    $record->transaction_id = $a->transaction_id;
                    $record->product_id = $product_id;
                    $record->comment = sprintf("%s: product_id:%d access_id:%d (%s/%s)", $this->getId(),
                        $product->pk(), $a->pk(), $dates['from'], $dates['to']);
                    try {
                        $record->save();
                    } catch (Am_Exception_Db_NotUnique $e) {}
                }
            }
        }
    }

    function onAccessAfterInsert(Am_Event $event)
    {
        $access = $event->getAccess();
        $product = $this->getDi()->productTable->load($access->product_id);
        $schedule_access = json_decode($product->data()->getBlob('schedule_access'), true);

        if ($schedule_access) {
            foreach ($schedule_access as $product_id => $dates)
            {
                if ($product->pk() == $product_id)
                    continue;

                if ($dates['from'] == self::START) {
                    $record = $this->getDi()->accessRecord;
                    $record->user_id = $access->user_id;
                    $record->begin_date = $this->getDi()->sqlDate;
                    $record->expire_date = $this->getExpire($this->getDi()->dateTime, $access->begin_date, $access->expire_date, $dates['to']);
                    $record->invoice_id = $access->invoice_id;
                    $record->invoice_public_id = $access->invoice_public_id;
                    $record->invoice_payment_id = $access->invoice_payment_id;
                    $record->transaction_id = $access->transaction_id;
                    $record->product_id = $product_id;
                    $record->comment = sprintf("%s: product_id:%d access_id:%d (%s/%s)", $this->getId(), $product->pk(), $access->pk(), $dates['from'], $dates['to']);
                    try {
                        $record->save();
                    } catch (Am_Exception_Db_NotUnique $e) {}
                } elseif (preg_match('/(^[0-9]+)-p$/', $dates['from'], $m)) {
                    $cnt = $this->getDi()->db->selectCell(<<<CUT
                            SELECT COUNT(DISTINCT invoice_payment_id)
                                FROM ?_invoice_payment
                                LEFT JOIN ?_invoice_item USING (invoice_id)
                                WHERE user_id = ?
                                    AND item_type='product'
                                    AND item_id=?;
CUT
                        , $access->user_id, $access->product_id);
                    if ($cnt == $m[1]) {
                        $record = $this->getDi()->accessRecord;
                        $record->user_id = $access->user_id;
                        $record->begin_date = $this->getDi()->sqlDate;
                        $record->expire_date = $this->getExpire($this->getDi()->dateTime, $access->begin_date, $access->expire_date, $dates['to']);
                        $record->invoice_id = $access->invoice_id;
                        $record->invoice_public_id = $access->invoice_public_id;
                        $record->invoice_payment_id = $access->invoice_payment_id;
                        $record->transaction_id = $access->transaction_id;
                        $record->product_id = $product_id;
                        $record->comment = sprintf("%s: product_id:%d access_id:%d (%s/%s)", $this->getId(), $product->pk(), $access->pk(), $dates['from'], $dates['to']);
                        try {
                            $record->save();
                        } catch (Am_Exception_Db_NotUnique $e) {}
                    }
                }
            }
        }
    }

    function onSubscriptionDeleted(Am_Event_SubscriptionDeleted $event)
    {
        $user = $event->getUser();
        $product = $event->getProduct();
        $access = $this->getDi()->accessTable->findFirstBy([
            'user_id' => $user->pk(),
            'product_id' => $product->pk()
        ], null, null, 'expire_date DESC');

        if (!$access) return; //access record was removed

        $schedule_access = json_decode($product->data()->getBlob('schedule_access'), true);
        if ($schedule_access) {
            foreach ($schedule_access as $product_id => $dates)
            {
                if ($product->pk() == $product_id)
                    continue;

                if ($dates['from'] != self::EXPIRE) continue;

                $record = $this->getDi()->accessRecord;
                $record->user_id = $access->user_id;
                $record->begin_date = $this->getDi()->sqlDate;
                $record->expire_date = $this->getExpire($this->getDi()->dateTime, $access->begin_date, $access->expire_date, $dates['to']);
                $record->invoice_id = $access->invoice_id;
                $record->invoice_public_id = $access->invoice_public_id;
                $record->invoice_payment_id = $access->invoice_payment_id;
                $record->transaction_id = $access->transaction_id;
                $record->product_id = $product_id;
                $record->comment = sprintf("%s: product_id:%d access_id:%d (%s/%s)", $this->getId(), $product->pk(), $access->pk(), $dates['from'], $dates['to']);
                try {
                    $record->save();
                } catch (Am_Exception_Db_NotUnique $e) {}
            }
        }
    }

    function getExpire(DateTime $dateNow, $begin_date, $expire_date, $end)
    {
        if ($end == self::FOREVER)
            return Am_Period::MAX_SQL_DATE;
        if ($end == self::EXPIRE)
            return $expire_date;
        [$count, $unit] = explode('-', $end);
        preg_match('/(m|d|w)([0-9]{0,2})/', $unit, $matches);

        $date = clone $dateNow;
        $time = amstrtotime($begin_date);
        $date->setDate(date('Y', $time), date('m', $time), date('d', $time));

        switch ($matches[1])
        {
            case 'd':
                $date->modify("+{$count} days");
                return $date->format('Y-m-d');
            case 'w':
                $now = $date->format('w');
                $search = $matches[2];
                $d = ($search - $now + 7) % 7;
                $date->modify("+{$d} days");
                $d = ($count - 1) * 7;
                $date->modify("+{$d} days");
                return $date->format('Y-m-d');
            case 'm':
                $now = $date->format('d');
                $m = $now < $matches[2] ? $count - 1 : $count;
                $date->modify("+{$m} months");
                return $date->format('Y-m-' . $matches[2]);
            default:
                throw new Am_Exception_FatalError(sprintf('Incorrect interval point format [%s] in %s::%s',
                        $end, __CLASS__, __METHOD__));
        }
    }

    /**
     * calculate interval of access start dates applicable for rule
     *
     * @param DateTime $dateNow
     * @param string $start
     * @param int $product_id
     * @return array return empty interval in case of we should not check this rule
     */
    function getSuitableStartInterval(DateTime $dateNow, $start, $product_id)
    {
        if ($this->isEventBased($start)) {
            return $this->returnEmptyInterval();
        }

        [$count, $unit] = explode('-', $start);
        preg_match('/(m|d|w)([0-9]{0,2})/', $unit, $matches);
        $start = clone $dateNow;

        switch ($matches[1])
        {
            case 'd':
                $dStart = $count - 1;
                $start->modify("-{$dStart} days");
                return [
                    $start->format('Y-m-d'),
                    $start->format('Y-m-d')
                ];
            case 'w':
                $w = $matches[2];
                if ($start->format('w') != $w)
                    return $this->returnEmptyInterval();
                $dStart = 7 * $count - 1;
                $start->modify("-{$dStart} days");
                $stop = clone $start;
                $stop->modify("+6 days");
                return [
                    $start->format('Y-m-d'),
                    $stop->format('Y-m-d')
                ];
            case 'm':
                $d = $matches[2];
                if ($start->format('d') != $d)
                    return $this->returnEmptyInterval();
                $mStart = $count;
                $start->modify("-{$mStart} months");
                $start->modify("+1 days");
                $stop = clone $start;
                $stop->modify("+1 month");
                $stop->modify("-1 days");
                return [
                    $start->format('Y-m-d'),
                    $stop->format('Y-m-d')
                ];
            default:
                throw new Am_Exception_Configuration(sprintf('Incorrect interval point format [%s] in %s::%s for product #%d',
                        $start, __CLASS__, __METHOD__, $product_id));
        }
    }

    protected function returnEmptyInterval()
    {
        //start is greater than end. it is always empty interval
        return ['2012-02-03', '2012-02-02'];
    }
}

class Am_Form_Element_ScheduleAccess extends HTML_QuickForm2_Element
{
    /**
     * @var HTML_QuickForm2_Element_InputHidden
     */
    protected $hidden;

    public function __construct($name = null, $attributes = null, $data = null)
    {
        $attributes = [
            'class' => 'schedule-access',
            'data-option-product' => json_encode($this->getProductOptions()),
            'data-option-unit-from' => json_encode($this->getUnitFromOptions()),
            'data-option-unit-to' => json_encode($this->getUnitToOptions()),
            'data-option-unit-all' => json_encode($this->getUnitAllOptions()),
        ];

        $this->hidden = new HTML_QuickForm2_Element_InputHidden($name, $attributes, $data);

        parent::__construct($name);
    }

    protected function getProductOptions()
    {
        static $options;
        if (!$options)
            $options = Am_Di::getInstance()->productTable->getOptions();
        return $options;
    }

    protected function getUnitFromOptions()
    {
        return [
            Am_Plugin_ScheduleAccess::START => 'start',
            Am_Plugin_ScheduleAccess::EXPIRE => 'expire',
            'Day' =>
                [
                    'd' => 'Day'
                ],
            'Weekday' =>
                [
                    'w1' => 'Monday',
                    'w2' => 'Tuesday',
                    'w3' => 'Wednesday',
                    'w4' => 'Thursday',
                    'w5' => 'Friday',
                    'w6' => 'Saturday',
                    'w0' => 'Sunday',
                ],
            'Monthday' =>
                [
                    'm1' => '1-st day of Month',
                    'm15' => '15-th day of Month',
                ],
            'Payment' =>
                [
                    'p' => 'Payment'
                ],
        ];
    }

    protected function getUnitToOptions()
    {
        return [
            Am_Plugin_ScheduleAccess::EXPIRE => 'expire',
            Am_Plugin_ScheduleAccess::FOREVER => 'forever',
            'Day' =>
                [
                    'd' => 'Day'
                ],
            'Weekday' =>
                [
                    'w1' => 'Monday',
                    'w2' => 'Tuesday',
                    'w3' => 'Wednesday',
                    'w4' => 'Thursday',
                    'w5' => 'Friday',
                    'w6' => 'Saturday',
                    'w0' => 'Sunday',
                ],
            'Monthday' =>
                [
                    'm1' => '1-st day of Month',
                    'm15' => '15-th day of Month',
                ],
        ];
    }

    protected function getUnitAllOptions()
    {
        return [
            Am_Plugin_ScheduleAccess::START => 'start',
            Am_Plugin_ScheduleAccess::EXPIRE => 'expire',
            Am_Plugin_ScheduleAccess::FOREVER => 'forever',
            'Day' =>
                [
                    'd' => 'Day'
                ],
            'Weekday' =>
                [
                    'w1' => 'Monday',
                    'w2' => 'Tuesday',
                    'w3' => 'Wednesday',
                    'w4' => 'Thursday',
                    'w5' => 'Friday',
                    'w6' => 'Saturday',
                    'w0' => 'Sunday',
                ],
            'Monthday' =>
                [
                    'm1' => '1-st day of Month',
                    'm15' => '15-th day of Month',
                ],
            'Payment' =>
                [
                    'p' => 'Payment'
                ],
        ];
    }

    public function getJs()
    {
        return <<<'CUT'
jQuery(function(){
    jQuery('.schedule-access').each(function(){
        if (jQuery(this).data('schedule-access')) return;
        jQuery(this).data('schedule-access', 1);
        var $this = jQuery(this);

        $this.data('value', $this.val() ? jQuery.evalJSON($this.val()) : {});

        //create product select and populate it with available options
        var $productSel = jQuery('<select></select>');
        $productSel.append('<option value="__special_offer">Please select an item</a>');
        populateSelect($productSel, $this.data('option-product'));

        var $wrapper = $this.wrap('<div></div>').closest('div');
        $wrapper.append('<h3>Schedule Subscription to:</h3>');
        $wrapper.append($productSel);
        
        jQuery($productSel).select2(p = {
            minimumResultsForSearch : 10,
        }).data('select2-option', p);

        $productSel.change(function(){
            if (jQuery(this).val() != '__special_offer') {
                var ac_id = jQuery(this).val();
                jQuery(this).find('option:selected').prop('disabled', 'disabled');
                jQuery(this).val('__special_offer');
                jQuery(this).change();
                addAccessLine(ac_id, 'start', 'expire');
            }
        })

        jQuery.each($this.data('value'), function(index, el){
            addAccessLine(index, el.from, el.to);
            $productSel.find('option[value=' + index + ']').prop('disabled', 'disabled');
        })

        // --- only functions below this line ----

        function updateAccess(ac_id, point_from, point_to) {
            $this.data('value')[ac_id] = {from : point_from, to : point_to};
            updateValue($this.data('value'));
        }

        function updateAccessFrom(ac_id, point) {
            $this.data('value')[ac_id]['from'] = point;
            updateValue($this.data('value'));
        }

        function updateAccessTo(ac_id, point) {
            $this.data('value')[ac_id]['to'] = point;
            updateValue($this.data('value'));
        }

        function deleteAccess(ac_id) {
            delete $this.data('value')[ac_id];
            updateValue($this.data('value'));
        }

        function updateValue(value) {
            $this.val(jQuery.toJSON(value));
        }

        function expandIntervalPointDef(interval_point) {
            switch (interval_point) {
                case 'start':
                    return [NaN, 'start'];
                case 'expire':
                    return [NaN, 'expire'];
                case 'forever' :
                    return [NaN, 'forever'];
                default :
                    return interval_point.split('-', 2);
            }
        }

        function collapseIntervalPointDef(count, unit) {
            switch (unit) {
                case 'start':
                case 'expire':
                case 'forever':
                    return unit;
                default :
                    return (parseInt(count) || 1) + '-' + unit;
            }
        }

        function getIntervalDesc(interval_point) {
            var interval_point = expandIntervalPointDef(interval_point);
            var count = interval_point[0];
            var unit = interval_point[1];
            if (!count) return unit;

            return count + ' - ' + findUnitTitle(unit, $this.data('option-unit-all'));
        }

        function findUnitTitle(unit, options) {
            if (options[unit]) return options[unit];
            var $res = null;
            jQuery.each(options, function(index, el) {
                if (jQuery.isPlainObject(el)) {
                    $title = findUnitTitle(unit, el);
                    $res = $title ? $title : $res;
                }
            })
            return $res; //did not find anything
        }

        function createIntervalPointEditor(ac_id, interval_point, isStart) {
            var $intervalPoint = jQuery('<span></span>');
            var point = expandIntervalPointDef(interval_point);
            var count = point[0];
            var unit = point[1];

            var $countEl = jQuery('<input type="text" value="1" size="4" />').css('display', 'inline');
            count ? $countEl.val(count) : $countEl.css('display', 'none');

            var $unitEl = jQuery('<select></select>').css('display', 'inline');
            var $options = isStart ? $this.data('option-unit-from') : $this.data('option-unit-to');

            populateSelect($unitEl, $options);

            $unitEl.val(unit);

            $unitEl.change(function() {
                var interval_point = expandIntervalPointDef(collapseIntervalPointDef($countEl.val(), $unitEl.val()));
                var count = interval_point[0];
                var unit = interval_point[1];
                count ? $countEl.css('display', 'inline') : $countEl.css('display', 'none');
            })

            var $container = jQuery('<span></span>').css({'font-size' : '8pt', 'display' : 'inline'}).append($countEl).append($unitEl).css('display', 'none');

            var $linkHtml = jQuery('<a href="javascript:;" class="local"></a>').text(getIntervalDesc(interval_point));

            $linkHtml.click(function(){
                jQuery(this).hide();
                $container.css('display', 'inline');
                $container.bind('outerClick', function() {
                    jQuery(this).unbind('outerClick');
                    jQuery(this).css('display', 'none');
                    var point = collapseIntervalPointDef($countEl.val(), $unitEl.val());
                    $linkHtml.text(getIntervalDesc(point));
                    if (isStart) {
                        updateAccessFrom(ac_id, point);
                    } else {
                        updateAccessTo(ac_id, point);
                    }
                    $linkHtml.show();
                })
                return false;
            })

            $intervalPoint.append($linkHtml).append($container);
            return $intervalPoint;
        }

        function addAccessLine (ac_id, point_from, point_to) {
            var title = $this.data('option-product')[ac_id];

            var $a = jQuery('<a href="javascript:;" class="am-link-del">&#10005;</a>').css('text-decoration', 'none').click(function(){
                jQuery(this).closest('div').remove();
                $productSel.find('option[value=' + ac_id + ']').prop('disabled', '');
                deleteAccess(ac_id);
            })

            var $access = jQuery('<div></div>').css('margin-top', '0.2em').
                    append($a).
                    append(' ').
                    append(jQuery('<strong></strong>').text(title)).
                    append(' ').
                    append('from').
                    append(' ').
                    append(createIntervalPointEditor(ac_id, point_from, true)).
                    append(' ').
                    append('to').
                    append(' ').
                    append(createIntervalPointEditor(ac_id, point_to, false))

            $wrapper.append($access);
            updateAccess(ac_id, point_from, point_to);
        }

        function populateSelect ($sel, options) {
            jQuery.each(options, function(index, el) {
                if (jQuery.isPlainObject(el)) {
                    var $opt = jQuery('<optgroup></optgroup>').attr('label', index);
                    jQuery.each(el, function(index, el){
                        var $option = jQuery('<option></option>');
                        $option.attr('value', index);
                        $option.text(el);
                        $opt.append($option)
                    })

                    $sel.append($opt);
                } else {
                    var $option = jQuery('<option></option>');
                    $option.attr('value', index);
                    $option.text(el);

                    $sel.append($option);
                }
            })
        }
    })
});
CUT;
    }

    public function getType()
    {
        return 'schedule-access';
    }

    public function __toString()
    {
        return $this->hidden->__toString();
    }

    public function getRawValue()
    {
        return $this->hidden->getRawValue();
    }

    public function setValue($value)
    {
        return $this->hidden->setValue($value);
    }

    public function updateValue()
    {
        $this->hidden->setContainer($this->getContainer());
        $this->hidden->updateValue();
    }

    public function render(HTML_QuickForm2_Renderer $renderer)
    {
        $renderer->getJavascriptBuilder()->addElementJavascript($this->getJs());
        return parent::render($renderer);
    }
}