<?php
/**
 * @table paysystems
 * @id cryptonator
 * @title Mycryptocheckout
 * @desc Accept cryptocurrency payments in your webshop with no middleman or transaction fees. Payments go directly from the customer to any wallet you choose
 * @visible_link https://mycryptocheckout.com/
 * @logo_url mycryptocheckout.png
 * @recurring none
 * @am_payment_api 6.0
 */
class Am_Paysystem_Mycryptocheckout extends Am_Paysystem_Abstract
{
    const PLUGIN_STATUS = self::STATUS_BETA;
    const PLUGIN_REVISION = '6.3.30';

    protected $defaultTitle = 'Mycryptocheckout';
    protected $defaultDescription = 'paid by bitcoins';

    const ENDPOINT = 'https://api.mycryptocheckout.com/v2/';

    const DEFAULT_PAYMENT_TIMEOUT = 2;

    public function getRecurringType()
    {
        return self::REPORTS_NOT_RECURRING;
    }

    public function _initSetupForm(Am_Form_Setup $form)
    {
        if (isset($_GET['connect'])) {
            return $this->connect();
        }

        if (isset($_GET['retrieve'])) {
            return $this->doRetrieve();
        }

        $account = $this->account();
        if (!isset($account['payments_left'])) {
            $form->addHtml()
                ->setHtml('Account is not connected');

            $url = $this->getDi()->url("admin-setup/{$this->getId()}?connect");
            $form->addHtml()
                ->setHtml(<<<CUT
<a href="$url">Connect</a>
CUT
);
        } else {
            $url = $this->getDi()->url("admin-setup/{$this->getId()}?retrieve");
            $form->addHtml()
                ->setHtml(<<<CUT
<a href="$url">Refresh Account Data</a>
CUT
                );

            $form->addHtml()
                ->setLabel('Payments remaining this month')
                ->setHtml($account['license_valid'] ? 'Unlimited' : Am_Html::escape((string)$account['payments_left']));

            if (!empty($account['license_valid_until'])) {
                $form->addHtml()
                    ->setLabel('License Expires')
                    ->setHtml(Am_Html::escape(amDatetime((int)$account['license_valid_until'])));
            }

            $purchase_link = Am_Html::escape('https://mycryptocheckout.com/pricing/?' . http_build_query(['domain' => base64_encode($this->getPluginUrl('msg'))]));
            $form->addHtml()
                ->setLabel('Purchase/Extend a license for unlimited payments')
                ->setHtml(<<<CUT
<a href="$purchase_link">Purchase</a>
CUT
);
            $form->addHtml()
                ->setLabel('Cryptocurrency exchange rates updated')
                ->setHtml(amDatetime($account['virtual_exchange_rates']['timestamp']));
            $form->addHtml()
                ->setLabel('Physical currency exchange rates updated')
                ->setHtml(amDatetime($account['physical_exchange_rates']['timestamp']));

            $url = $this->getDi()->url("payment/{$this->getId()}/wallet");
            $form->addHtml()
                ->setLabel('Wallets')
                ->setHtml(<<<CUT
<a href="{$url}">manage</a>
CUT
);
            $form->addInteger('payment_amount_spread')
                ->setLabel("Payment amount spread\n"
                    . "if you are anticipating several purchases a second with the"
                    . " same currency, increase this amount to 100 or more to help"
                    . " prevent duplicate amount payments by slightly increasing the"
                    . " payment at random");

            $form->addInteger('timeout_hours', ['placeholder' => self::DEFAULT_PAYMENT_TIMEOUT])
                ->setLabel("Payment timeout\nHow many hours to wait for the payment to come through before marking the order as abandoned");
        }
    }

    /**
     * It is workaround. In order to complete configuration we need that plugin was already configured (directAction)
     * so we use isNotAcceptableForInvoice to prevent usage of plugin in non-configured state
     */
    public function _isConfigured()
    {
        $_ = json_decode($this->getDi()->store->getBlob("{$this->getId()}.wallet") ?: '[]', true);
        $account = $this->account();

        return isset($account['payments_left']) && $account['payments_left'] > 0 && count($_) > 0;
    }

    public function isNotAcceptableForInvoice(Invoice $invoice)
    {
        if (!$this->_isConfigured()) {
            return ['Plugin configuration is not finished yet'];
        }

        return parent::isNotAcceptableForInvoice($invoice);
    }

    public function getSupportedCurrencies()
    {
        $account = $this->account();

        return array_keys($account['physical_exchange_rates']['rates']);
    }

    public function _process($invoice, $request, $result)
    {
        $action = new Am_Paysystem_Action_Redirect($this->getPluginUrl('currency', ['id' => $invoice->getSecureId($this->getId())]));
        $result->setAction($action);
    }

    function convert($amount, $fiat_currency, $crypto_currency)
    {
        $account = $this->account();

        $fiat_exchange_rate = $account['physical_exchange_rates']['rates'][$fiat_currency];
        $usd_amount = $amount / $fiat_exchange_rate;

        $crypto_exchange_rate = $account['virtual_exchange_rates']['rates'][$crypto_currency];
        $crypto_amount = $usd_amount * $crypto_exchange_rate;

        $decimal_precision = $account['currency_data'][$crypto_currency]['decimal_precision'] ?? 8;
        return sprintf("%.{$decimal_precision}f", $crypto_amount);
    }

    function crypto_options()
    {
        $account = $this->account();
        return array_map(function($_) {return $_['name'];}, $account['currency_data']);
    }

    function my_crypto_options()
    {
        $op = $this->crypto_options();
        $_ = json_decode($this->getDi()->store->getBlob("{$this->getId()}.wallet"), true);

        $result = [];
        foreach ($_ as $item) {
            $result[$item['currency_id']] = $op[$item['currency_id']];
        }

        return $result;
    }

    function crypto_desc($currency_id)
    {
        $account = $this->account();
        return $account['currency_data'][$currency_id];
    }

    function crypto_currency_address($currency_id)
    {
        $_ = json_decode($this->getDi()->store->getBlob("{$this->getId()}.wallet"), true);
        foreach ($_ as $item) {
            if ($item['currency_id'] == $currency_id) return $item['address'];
        }
    }

    function crypto_currency_monero_private_view_key($currency_id)
    {
        $_ = json_decode($this->getDi()->store->getBlob("{$this->getId()}.wallet"), true);
        foreach ($_ as $item) {
            if ($item['currency_id'] == $currency_id) return $item['monero_private_view_key'];
        }
    }

    public function find_next_available_amount($amount, $crypto_currency)
    {
        while (!$this->is_payment_amount_available($amount, $crypto_currency)) {
            $amount = $this->increment_payment_amount($amount, $crypto_currency);
        }

        return $amount;
    }

    function increment_payment_amount($amount, $crypto_currency)
    {
        $account = $this->account();
        $decimal_precision = $account['currency_data'][$crypto_currency]['decimal_precision'] ?? 8;

        $amount = sprintf("%.{$decimal_precision}f", $amount + pow(10, -1 * $decimal_precision));

        return $amount;
    }

    public function is_payment_amount_available($amount, $crypto_currency)
    {
        $account = $this->account();
        $pa = $account['payment_amounts'];

        return empty($pa[$crypto_currency]) || !isset($pa[$crypto_currency][(string) $amount]);
    }

    public function directAction(Am_Mvc_Request $request, Am_Mvc_Response $response, array $invokeArgs)
    {
        switch ($request->getActionName()) {
            case 'retrieve' :
                $this->actionRetrieve($request, $response);
                break;
            case 'check' :
                $this->actionCheck($request, $response);
                break;
            case 'wallet':
                $this->actionWallet($request, $response);
                break;
            case 'msg':
                $this->process_messages($request, $response);
                break;
            case 'currency':
                $this->actionCurrency($request, $response);
                break;
            case 'instruction':
                $this->actionInstruction($request, $response);
                break;
            default:
                parent::directAction($request, $response, $invokeArgs);
        }
    }

    function actionCheck($request, $response)
    {
        if (!$invoice = $this->getDi()->invoiceTable->findBySecureId($request->getParam('id'), 'CHECK-STATUS')) {
            throw new Am_Exception_InputError;
        }

        if ($invoice->status <> Invoice::PENDING) {
            $status = 'ok';
        } elseif (!$invoice->data()->get("{$this->getId()}.payment_id")) {
            $status = 'cancel';
        } else {
            $status = 'pending';
        }

        return $this->getDi()->response->ajaxResponse(['status' => $status]);
    }

    function onDaily(Am_Event $e)
    {
        //update currency rates
        $this->retrieve();
        $this->getDi()->db->query("DELETE FROM ?_invoice_log WHERE paysys_id=? AND tm<?", $this->getId(), sqlTime('-7days'));
    }

    function retrieve()
    {
        $retrieve_key = md5($this->getDi()->security->randomString(10));
        $this->getDi()->store->set("{$this->getId()}.retrieve_key", $retrieve_key);

        $req = new Am_HttpRequest(self::ENDPOINT . 'account/retrieve', Am_HttpRequest::METHOD_POST);
        $req->addPostParameter([
            'domain' => base64_encode($this->getPluginUrl('msg')),
            'retrieve_key' => $retrieve_key,
        ]);

        Zend_Session::writeClose();

        $res = $req->send();
        $r = json_decode($res->getBody(), true);
        $this->logMessage($r);
    }

    function connect()
    {
        $account = $this->account();
        if (isset($account['payments_left'])) {
            throw new Am_Exception_InputError('Already Connected');
        }

        $this->retrieve();

        Am_Mvc_Response::redirectLocation($this->getDi()->url("admin-setup/{$this->getId()}", false));
    }

    function doRetrieve()
    {
        $this->retrieve();

        Am_Mvc_Response::redirectLocation($this->getDi()->url("admin-setup/{$this->getId()}", false));
    }

    function actionRetrieve($request, $response)
    {
        if (!$this->getDi()->authAdmin->getUser()
            || !$this->getDi()->authAdmin->getUser()->hasPermission(Am_Auth_Admin::PERM_SETUP)) {

            throw new Am_Exception_AccessDenied;
        }

        $this->retrieve();

        Am_Mvc_Response::redirectLocation($this->getDi()->url("admin-setup/{$this->getId()}", false));
    }

    function actionWallet($request, $response)
    {
        if (!$this->getDi()->authAdmin->getUser()
            || !$this->getDi()->authAdmin->getUser()->hasPermission(Am_Auth_Admin::PERM_SETUP)) {

            throw new Am_Exception_AccessDenied;
        }

        $_ = json_decode($this->getDi()->store->getBlob("{$this->getId()}.wallet") ?: '[]', true);

        $ds = new Am_Grid_DataSource_MycryptocheckoutWallet($_, $this->getId());
        $grid = new Am_Grid_Editable('_mcc_w', 'Wallets', $ds, $request, $this->getDi()->view);
        $grid->addField('currency_id', 'Currency');
        $grid->addField('address', 'Address');

        $grid->addCallback(Am_Grid_Editable::CB_VALUES_FROM_FORM, function(& $v, $r) {
            unset($v['_csrf']);
        });

        $grid->actionAdd(new Am_Grid_Action_Sort_MycryptocheckoutWallet);

        $grid->setForm([$this, 'createForm']);

        $grid->runWithLayout('admin/layout.phtml');
    }

    function createForm()
    {
        $form = new Am_Form_Admin();
        $form->addSelect('currency_id', ['class' => 'am-combobox'])
            ->setLabel('Currency')
            ->loadOptions($this->crypto_options())
            ->addRule('required');
        $form->addText('address', ['class' => 'am-el-wide'])
            ->setLabel('Address')
            ->addRule('required');
        $form->addText('monero_private_view_key', ['class' => 'am-el-wide'])
            ->setLabel("Monero private view key\nYour private view key that is used to see the amounts in private transactions to your wallet.");

        $form->addScript()
            ->setScript(<<<CUT
jQuery(function(){
    jQuery('[name=currency_id]').change(function(){
        jQuery('[name=monero_private_view_key]').closest('.am-row').toggle(jQuery(this).val()=='XMR');
    }).change();
})
CUT
);
        return $form;
    }

    function actionCurrency($request, $response)
    {
        if (!$invoice = $this->getDi()->invoiceTable->findBySecureId($request->getParam('id'), $this->getId())) {
            throw new Am_Exception_InputError;
        }

        if ($invoice->data()->get("{$this->getId()}.payment_id")) {
            Am_Mvc_Response::redirectLocation($this->getPluginUrl('instruction') . '?id=' . $invoice->getSecureId($this->getId()));
        }

        $options = $this->my_crypto_options();
        if (count($options) == 1) {
            Am_Mvc_Response::redirectLocation($this->getPluginUrl('instruction', [
                    'id' => $invoice->getSecureId($this->getId()),
                    'crypto_currency' => key($options),
                ]
            ));
        }

        $form = new Am_Form();
        $form->setAction($this->getPluginUrl('instruction'));
        $form->addAdvRadio('crypto_currency')
            ->setLabel('Please select a currency')
            ->loadOptions($this->my_crypto_options())
            ->addRule('required');
        $form->addHidden('id')->setValue($request->getParam('id'));
        $form->addSaveButton('Next');

        $v = $this->getDi()->view;
        $v->title = "Crypto Currency";
        $v->content = $form;
        $v->display('layout.phtml');
    }

    function actionInstruction($request, $response)
    {
        if (!$invoice = $this->getDi()->invoiceTable->findBySecureId($request->getParam('id'), $this->getId())) {
            throw new Am_Exception_InputError;
        }

        if ($invoice->data()->get("{$this->getId()}.payment_id")) {
            return $this->payment_box($invoice);
        }

        if (!$crypto_currency = $request->getParam('crypto_currency')) {
            throw new Am_Exception_InputError;
        }

        $amount = $this->convert($invoice->first_total, $invoice->currency, $crypto_currency);
        $amount = $this->find_next_available_amount($amount, $crypto_currency);

        $amounts = [$amount];
        $spread = (int)$this->getConfig('payment_amount_spread');
        for ($i = 0; $i < $spread ; $i++ ) {
            $amount = $this->increment_payment_amount($amount, $crypto_currency);
            $amount = $this->find_next_available_amount($amount, $crypto_currency);
            $amounts[] = $amount;
        }

        $amount = $amounts[ array_rand( $amounts ) ];

        if (!$to = $this->crypto_currency_address($crypto_currency)) {
            throw new Am_Exception_InputError;
        }

        $req = new Am_HttpRequest(self::ENDPOINT . 'payment/add', Am_HttpRequest::METHOD_POST);

        $data = [
            'public_id' => $invoice->public_id,
        ];

        if ($crypto_currency == 'XMR') {
            $data['monero_private_view_key'] = $this->crypto_currency_monero_private_view_key($crypto_currency);
        }

        $req->addPostParameter([
            'domain' => $this->getPluginUrl('msg'),
            'domain_key' => $this->getDi()->store->get("{$this->getId()}.domain_key"),
            'payment' => [
                'amount' => $amount,
                'confirmations' => 1,
                'created_at' => $created_at = time(),
                'currency_id' => $crypto_currency,
                'timeout_hours' => $timeout_hours = $this->getConfig('timeout_hours', self::DEFAULT_PAYMENT_TIMEOUT),
                'to' => $to,
                'data' => json_encode($data)
            ]
        ]);
        $res = $req->send();
        $this->log($req, $res, 'Payment');

        $r = json_decode($res->getBody(), true);
        if (empty($r['payment_id'])) {
            throw new Am_Exception_InternalError;
        }

        $invoice->data()->set("{$this->getId()}.payment_id", $r['payment_id']);
        $invoice->data()->set("{$this->getId()}.currency", $crypto_currency);
        $invoice->data()->set("{$this->getId()}.amount", $amount);
        $invoice->data()->set("{$this->getId()}.address", $to);
        $invoice->data()->set("{$this->getId()}.timeout_hours", $timeout_hours);
        $invoice->data()->set("{$this->getId()}.created_at", $created_at);
        $invoice->save();

        Am_Mvc_Response::redirectLocation($this->getPluginUrl('instruction') . '?id=' . $invoice->getSecureId($this->getId()));
    }

    function payment_box($invoice)
    {
        $this->getDi()->view->headScript()
            ->appendFile(
                'https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js',
                'text/javascript',
                [
                    'integrity' => 'sha512-CNgIRecGo7nphbeZ04Sc13ka07paqdeTu0WR1IM4kNcpmBAUSHSQX0FslNhTDadL4O5SAGapGt4FodqL8My0mA==',
                    'crossorigin' => 'anonymous',
                    'referrerpolicy' => 'no-referrer',
                ]);

        $this->invoice = $invoice;

        $currency = $invoice->data()->get("{$this->getId()}.currency");
        $amount = $invoice->data()->get("{$this->getId()}.amount");
        $address = $invoice->data()->get("{$this->getId()}.address");
        $created_at = $invoice->data()->get("{$this->getId()}.created_at");
        $timeout_hours = $invoice->data()->get("{$this->getId()}.timeout_hours");

        $start = $created_at + $timeout_hours * 60 * 60 - time();

        $desc = $this->crypto_desc($currency);

        $qr_data = $address;
        if (isset($desc['qr_code'])) {
            $qr_data = str_replace(['[MCC_TO]', '[MCC_AMOUNT]'], [$address, $amount], $desc['qr_code']);
        }

        $qr_data = json_encode($qr_data);
        $qr_code = <<<CUT
<div style="padding: 1em"><div id="qrcode" style="width:150px; margin: 0 auto"></div></div>
<script type="text/javascript">
const qrcode = new QRCode(document.getElementById("qrcode"), {
	text: $qr_data,
	width: 150,
	height: 150,
	correctLevel : QRCode.CorrectLevel.L
});
</script>
CUT;

        $descJson = json_encode([
            'to' => $address,
            'amount' => $amount,
            'currency' => $desc,
        ]);

        $check_url = json_encode($this->getPluginUrl('check') . '?' . http_build_query([
            'id' => $invoice->getSecureId('CHECK-STATUS')
        ]));

        $return_url = json_encode($this->getReturnUrl());
        $cancel_url = json_encode($this->getCancelUrl());
        $return_url_html = Am_Html::escape($this->getReturnUrl());
        $cancel_url_html = Am_Html::escape($this->getCancelUrl());

        $content = <<<CUT
<style type="text/css">
.data-wrapper {
    border: 1px solid #eee;
    border-radius: 3px;
    padding: .2em .5em;
    display: inline-block;
}
.data-amount,
.data-address {
    margin-bottom: .5em;
}
.am-payment-box {
 max-width: 500px;
 border: 1px solid #eee;
 padding: 1em;;
}
.am-payment-box h3 {
   margin-bottom: 1em;
}

/* metamask */

.metamask_payment {
  display: inline-block;
  margin-top: 5px;
  margin-bottom: 10px;
  width: 250px;
  height: 73px;
  background-size: cover;
  max-width: 100%;
  background-repeat: no-repeat;
  background-position: center center;
  cursor: pointer;
  box-shadow: 0 4px 4px 0 rgba(0, 0, 0, 0.1), 0 6px 10px 0 rgba(0, 0, 0, 0.19);
  background-image: url("data:image/svg+xml;base64,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");
}

.metamask_payment:hover {
  box-shadow: 0 6px 8px 0 rgba(0, 0, 0, 0.24), 0 10px 20px 0 rgba(0, 0, 0, 0.19);
}

.metamask_payment:active {
  box-shadow: 0 4px 4px 0 rgba(0, 0, 0, 0.1), 0 6px 10px 0 rgba(0, 0, 0, 0.19);
}

.data-qr-timer-wrapper {
    margin-top:1em;
    display:flex;
}

.data-timer {
    text-align:center;
    margin-top:1.5em;
}

.data-timer-timer {
    color: white;
    border-radius: 3px;
    background: #313131;
    padding:.5em;
}

.am-payment-box .data-timer-text {
    margin-bottom:1em;
    line-height: 1.2em;
}

.data-qr {
    margin-right:2em;
}

</style>
<div class="am-payment-box">
<h3>To complete your order</h3>
<div class="data-amount">Send: <span class="data-wrapper"><strong class="am-copy-to-clipboard">$amount</strong></span> <strong>$currency</strong></div>
<div class="data-address">To: <span class="data-wrapper"><strong class="am-copy-to-clipboard">{$address}</strong></span></div>
<div class="data-qr-timer-wrapper">
    <div class="data-qr">$qr_code</div>
    <div class="data-timer">
        <p class="data-timer-text"><strong>Awaiting Payment</strong><br />(checked every 5 secs)</p>
        <p class="data-timer-counter"><span class="data-timer-timer am-countdown" data-start="{$start}" data-format="{h}:{m}:{s}" data-redirect="$cancel_url_html"></span></p>
    </div>
</div>
<div class="am-payment-box-buttons">

</div>
<p><a href="$return_url_html">Click Here if you have already completed transaction</a></p>
</div>
<script type="text/javascript">
    function checkAndRedirect() {
        jQuery.get($check_url, function(r) {
            switch(r['status']) {
                case 'ok':
                    window.location = $return_url;
                    break;
                case 'cancel':
                    window.location = $cancel_url;
                    break;
            }
        });
    }
    setInterval(checkAndRedirect, 5000);

    jQuery(function(){
        const data = $descJson;
       
		if ( typeof window.ethereum === 'undefined' || !ethereum.isMetaMask )
			return;

		window.web3 = new Web3(ethereum);

		// The data must support metamask.
		if ( typeof data.currency.supports === 'undefined' )
			return;

		var contractInstance = false;
		if (typeof data.currency.supports.metamask_abi !== 'undefined')
		{
			var contractInstance = new web3.eth.Contract(JSON.parse(data.currency.supports.metamask_abi), data.currency.contract);
		}

		if (contractInstance === false)
			if (typeof data.currency.supports.metamask_currency === 'undefined')
				return;

		const metamask = $('<div class="metamask_payment"></div>');
		metamask.appendTo( jQuery('.am-payment-box-buttons') );

		metamask.click( async function()
		{

			try {
				// Request account access if needed
				const accounts = await window.ethereum.request({ method: 'eth_requestAccounts' });

				var send_parameters = {
					'from' : accounts[0], // First available.
				};

				if ( contractInstance === false )
				{
					send_parameters[ 'to' ] = data.to;
					send_parameters[ 'value' ] = web3.utils.toHex(
						web3.utils.toWei(data.amount, data.currency.supports.metamask_currency)
					);
					await window.ethereum.request({
						method: 'eth_sendTransaction',
						params: [ send_parameters ],
					}, function (err, transactionHash ) {
						console.log( 'Error sending Eth via Metamask', err );
					});
				}
				else
				{
					var amount = data.amount;
					// If there is a divider, use it.
					if (typeof data.currency.divider !== 'undefined') {
						amount *= data.currency.divider;
					} else {
						amount = web3.utils.toWei(amount, data.currency.supports.metamask_currency);
					}

					amount = amount + "";

					if ( typeof data.currency.supports.metamask_gas !== 'undefined' )
					{
						var metamask_gas = data.currency.supports.metamask_gas;
						send_parameters[ 'gasPrice' ] = web3.utils.toWei( metamask_gas.price + '', 'gwei' );

						if ( typeof data.currency.supports.metamask_gas_limit !== 'undefined' ) {
							metamask_gas.limit = data.currency.supports.metamask_gas_limit;
						}
						send_parameters[ 'gas' ] = metamask_gas.limit + '';
					}

					contractInstance.methods
						.transfer(data.to, amount)
						.send( send_parameters );
				}
			} catch (error) {
				console.log( 'User denied account access', error );
			}
		});
    });
</script>
CUT;

        $v = $this->getDi()->view;
        $v->headScript()->appendFile('https://cdn.jsdelivr.net/npm/web3@latest/dist/web3.min.js');
        $v->title = 'Process Payment';
        $v->content = $content;
        $v->layoutNoMenu = true;
        $v->display('layout.phtml');
    }

    function account()
    {
        return json_decode($this->getDi()->store->getBlob("{$this->getId()}.account") ?: '[]', true);
    }

    function process_messages(Am_Mvc_Request $request, Am_Mvc_Response $response)
    {
        $body = $request->getRawBody();
        if (!$data = json_decode($body, true)) {
            return;
        }

        $retrieve_key = $this->getDi()->store->get("{$this->getId()}.retrieve_key");

        foreach ($data['messages'] as $message)
        {
            if ( $message['type'] == 'retrieve_account'
                && $retrieve_key
                && $message['retrieve_key'] == $retrieve_key) {

                $this->getDi()->store->setBlob("{$this->getId()}.account", json_encode($message['account']));
                $this->getDi()->store->set("{$this->getId()}.domain_key", $message['account']['domain_key']);
            }
        }

        // Request Validation
        if (empty($data['mycryptocheckout'])
            || $data['mycryptocheckout'] != $this->getDi()->store->get("{$this->getId()}.domain_key"))
        {
            $response->ajaxResponse(['result' => 'fail', 'message' => 'Invalid Domain Key']);
            return;
        }

        foreach ($data['messages'] as $message) {
            $this->logMessage($message);
            switch ($message['type']) {
                case 'retrieve_account':
                    // Already handled above.
                    break;
                case 'cancel_payment':
                    /** @var Invoice $inv */
                    $invoice = $this->getDi()->invoiceTable->findFirstByData("{$this->getId()}.payment_id", $message['payment']['payment_id']);
                    if ($invoice && $invoice->status == Invoice::PENDING) {
                        $invoice->data()->set("{$this->getId()}.payment_id", null);
                        $invoice->data()->set("{$this->getId()}.currency", null);
                        $invoice->data()->set("{$this->getId()}.amount", null);
                        $invoice->data()->set("{$this->getId()}.address", null);
                        $invoice->data()->set("{$this->getId()}.timeout_hours", null);
                        $invoice->data()->set("{$this->getId()}.created_at", null);
                        $invoice->save();
                    }
                    break;
                case 'payment_complete':
                    $invoice = $this->getDi()->invoiceTable->findFirstByData("{$this->getId()}.payment_id", $message['payment']['payment_id']);
                    if ($invoice && $invoice->status == Invoice::PENDING) {
                        $transaction = new Am_Paysystem_Transaction_Manual($this);
                        $transaction->setAmount($invoice->first_total)
                            ->setReceiptId($message['payment']['transaction_id']);
                        $invoice->addPayment($transaction);
                    }
                    break;
                case 'test_communication':
                    $response->ajaxResponse(['result' => 'ok', 'message' => date('Y-m-d H:i:s')]);
                    break;
                case 'update_account':
                    // Save our new account data.
                    $this->getDi()->store->setBlob("{$this->getId()}.account", json_encode($message['account']));
                    $this->getDi()->store->set("{$this->getId()}.domain_key", $message['account']['domain_key']);
                    break;
            }
        }

        $response->ajaxResponse(['result' => 'ok']);
    }

    public function createTransaction($request, $response, array $invokeArgs)
    {
        return null;
    }

    function logMessage($message)
    {
        //unset dictionary, it is too big to log
        unset($message['account']['physical_exchange_rates'], $message['account']['virtual_exchange_rates'], $message['account']['currency_data']);

        $l = $this->getDi()->invoiceLogRecord;
        $l->paysys_id = $this->getId();
        $l->title = 'Message';
        $l->add($message);
    }

    function log($req, $resp, $title)
    {
        $l = $this->getDi()->invoiceLogRecord;
        $l->paysys_id = $this->getId();
        $l->title = $title;
        $l->add($req);
        $l->add($resp);
    }
}

class Am_Grid_DataSource_MycryptocheckoutWallet extends Am_Grid_DataSource_Array
{
    protected $plugin_id;

    public function __construct(array $array, $plugin_id)
    {
        foreach ($array as $a) {
            $this->array[$this->getHash($a)] = $a;
        }
        $this->plugin_id = $plugin_id;
    }

    protected function persist($arr)
    {
        Am_Di::getInstance()->store->setBlob("{$this->plugin_id}.wallet", json_encode($arr));
    }

    protected function getHash($record)
    {
        return md5(serialize(get_object_vars((object)$record)));
    }

    public function createRecord()
    {
        $o = new stdClass;
        $o->currency_id = null;
        $o->address = null;
        return $o;
    }

    public function getDataSourceQuery()
    {
        return null;
    }

    public function _friendPersist()
    {
        $this->persist($this->array);
    }
}

class Am_Grid_Action_Sort_MycryptocheckoutWallet extends Am_Grid_Action_Sort_Abstract
{
    protected function setSortBetween($item, $after, $before)
    {
        $data = $this->grid->getDataSource()->_friendGetArray();

        $after = $after ? $after['id'] : null;
        $before = $before ? $before['id'] : null;
        $id = $item['id'];
        $it = $data[$id];

        $newData = [];

        foreach ($data as $k => $v) {
            if ($k == $id) continue;
            if ($k == $before) {
                $newData[$id] = $it;
            }
            $newData[$k] = $v;
            if ($k == $after) {
                $newData[$id] = $it;
            }
        }

        $this->grid->getDataSource()->_friendSetArray($newData);
        $this->grid->getDataSource()->_friendPersist();
    }
}